const express = require('express');
const bcrypt = require('bcryptjs');
const { body, validationResult } = require('express-validator');
const { User, Company } = require('../models');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Get all users with role-based filtering
router.get('/', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    let whereClause = {};
    
    // Role-based filtering
    if (req.user.role === 'admin') {
      // Admin can only see users from their company (including themselves)
      whereClause.company_id = req.user.company_id;
    } else if (req.user.role === 'user') {
      // User can only see themselves
      whereClause.id = req.user.id;
    }
    // Superadmin can see all users (no whereClause restriction)
    
    const { count, rows: users } = await User.findAndCountAll({
      where: whereClause,
      include: [
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      attributes: { exclude: ['password'] }, // Exclude password from response
      limit,
      offset,
      order: [['created_at', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: users,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching users:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get user statistics
router.get('/statistics', authenticateToken, async (req, res) => {
  try {
    let whereClause = {};
    
    // Role-based filtering for statistics
    if (req.user.role === 'admin') {
      whereClause.company_id = req.user.company_id;
    } else if (req.user.role === 'user') {
      whereClause.id = req.user.id;
    }
    // Superadmin can see all statistics (no whereClause restriction)
    
    const [totalUsers, superAdmins, admins, regularUsers] = await Promise.all([
      User.count({ where: whereClause }),
      User.count({ where: { ...whereClause, role: 'superadmin' } }),
      User.count({ where: { ...whereClause, role: 'admin' } }),
      User.count({ where: { ...whereClause, role: 'user' } })
    ]);

    res.json({
      totalUsers,
      superAdmins,
      admins,
      regularUsers
    });
  } catch (error) {
    console.error('Error fetching user statistics:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get single user
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const userId = parseInt(req.params.id);
    
    let whereClause = { id: userId };
    
    // Role-based access control
    if (req.user.role === 'admin') {
      // Admin can only view users from their company
      whereClause.company_id = req.user.company_id;
    } else if (req.user.role === 'user') {
      // User can only view themselves
      if (userId !== req.user.id) {
        return res.status(403).json({ message: 'You can only view your own profile' });
      }
    }
    // Superadmin can view any user
    
    const user = await User.findByPk(userId, {
      where: whereClause,
      include: [
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      attributes: { exclude: ['password'] }
    });

    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }

    res.json(user);
  } catch (error) {
    console.error('Error fetching user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Password complexity validation
const passwordComplexity = (password) => {
  const minLength = 8;
  const hasUpperCase = /[A-Z]/.test(password);
  const hasLowerCase = /[a-z]/.test(password);
  const hasNumbers = /\d/.test(password);
  const hasSpecialChar = /[!@#$%^&*(),.?":{}|<>]/.test(password);
  
  return {
    isValid: password.length >= minLength && hasUpperCase && hasLowerCase && hasNumbers && hasSpecialChar,
    minLength: password.length >= minLength,
    hasUpperCase,
    hasLowerCase,
    hasNumbers,
    hasSpecialChar
  };
};

// Create new user
router.post('/', authenticateToken, [
  body('name').notEmpty().withMessage('Name is required'),
  body('email').isEmail().withMessage('Valid email is required'),
  body('password').custom((value) => {
    const complexity = passwordComplexity(value);
    if (!complexity.isValid) {
      throw new Error('Password must be at least 8 characters with uppercase, lowercase, numbers, and special characters');
    }
    return true;
  }),
  body('role').isIn(['superadmin', 'admin', 'user']).withMessage('Invalid role'),
  body('company_id').optional().isInt().withMessage('Company ID must be an integer')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    // Only superadmin and admin can create users
    if (req.user.role === 'user') {
      return res.status(403).json({ message: 'You do not have permission to create users' });
    }
    
    const { name, email, password, role, company_id, ai_provider, ai_api_key } = req.body;
    
    // Role-based restrictions for user creation
    if (req.user.role === 'admin') {
      // Admin can only create users for their company and cannot create superadmin
      if (role === 'superadmin') {
        return res.status(403).json({ message: 'You cannot create superadmin users' });
      }
      if (company_id !== req.user.company_id) {
        return res.status(403).json({ message: 'You can only create users for your company' });
      }
    }
    
    // Check if user already exists
    const existingUser = await User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).json({ message: 'User with this email already exists' });
    }
    
    // Hash password
    const hashedPassword = await bcrypt.hash(password, 12);
    
    // Create user
    const user = await User.create({
      name,
      email,
      password: hashedPassword,
      role,
      company_id: req.user.role === 'admin' ? req.user.company_id : company_id,
      ai_provider,
      ai_api_key
    });
    
    // Return user without password
    const userResponse = await User.findByPk(user.id, {
      include: [
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      attributes: { exclude: ['password'] }
    });
    
    res.status(201).json(userResponse);
  } catch (error) {
    console.error('Error creating user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update user
router.put('/:id', authenticateToken, async (req, res) => {
  try {
    const userId = parseInt(req.params.id);
    
    // Find the user to update
    const userToUpdate = await User.findByPk(userId);
    if (!userToUpdate) {
      return res.status(404).json({ message: 'User not found' });
    }
    
    // Role-based restrictions for user updates
    if (req.user.role === 'user') {
      // User can only update themselves and cannot change role or company
      if (userId !== req.user.id) {
        return res.status(403).json({ message: 'You can only update your own profile' });
      }
      if (req.body.role || req.body.company_id) {
        return res.status(403).json({ message: 'You cannot change your role or company' });
      }
    } else if (req.user.role === 'admin') {
      // Admin cannot update superadmin users
      if (userToUpdate.role === 'superadmin') {
        return res.status(403).json({ message: 'You cannot modify superadmin users' });
      }
      // Admin can only update users from their company
      if (userToUpdate.company_id !== req.user.company_id) {
        return res.status(403).json({ message: 'You can only update users from your company' });
      }
      // Admin cannot promote users to superadmin
      if (req.body.role === 'superadmin') {
        return res.status(403).json({ message: 'You cannot promote users to superadmin' });
      }
      // Admin cannot change company_id
      if (req.body.company_id && req.body.company_id !== req.user.company_id) {
        return res.status(403).json({ message: 'You cannot change user company' });
      }
    }
    // Superadmin can update any user
    
    const { name, email, role, company_id, ai_provider, ai_api_key } = req.body;
    
    // Update user
    await userToUpdate.update({
      name,
      email,
      role: req.user.role === 'superadmin' ? role : userToUpdate.role,
      company_id: req.user.role === 'superadmin' ? company_id : userToUpdate.company_id,
      ai_provider,
      ai_api_key
    });
    
    // Return updated user without password
    const updatedUser = await User.findByPk(userId, {
      include: [
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      attributes: { exclude: ['password'] }
    });
    
    res.json(updatedUser);
  } catch (error) {
    console.error('Error updating user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete user
router.delete('/:id', authenticateToken, async (req, res) => {
  try {
    const userId = parseInt(req.params.id);
    
    // Find the user to delete
    const userToDelete = await User.findByPk(userId);
    if (!userToDelete) {
      return res.status(404).json({ message: 'User not found' });
    }
    
    // Prevent self-deletion
    if (userId === req.user.id) {
      return res.status(400).json({ message: 'You cannot delete your own account' });
    }
    
    // Role-based restrictions for user deletion
    if (req.user.role === 'user') {
      return res.status(403).json({ message: 'You do not have permission to delete users' });
    } else if (req.user.role === 'admin') {
      // Admin cannot delete superadmin users
      if (userToDelete.role === 'superadmin') {
        return res.status(403).json({ message: 'You cannot delete superadmin users' });
      }
      // Admin can only delete users from their company
      if (userToDelete.company_id !== req.user.company_id) {
        return res.status(403).json({ message: 'You can only delete users from your company' });
      }
    }
    // Superadmin can delete any user (except themselves)
    
    await userToDelete.destroy();
    
    res.json({ message: 'User deleted successfully' });
  } catch (error) {
    console.error('Error deleting user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Change user password
router.put('/:id/password', authenticateToken, async (req, res) => {
  try {
    const userId = parseInt(req.params.id);
    const { currentPassword, newPassword } = req.body;
    
    // Find the user
    const user = await User.findByPk(userId);
    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }
    
    // Role-based restrictions for password changes
    if (req.user.role === 'user') {
      // User can only change their own password and must provide current password
      if (userId !== req.user.id) {
        return res.status(403).json({ message: 'You can only change your own password' });
      }
      if (!currentPassword) {
        return res.status(400).json({ message: 'Current password is required' });
      }
      // Verify current password
      const isCurrentPasswordValid = await bcrypt.compare(currentPassword, user.password);
      if (!isCurrentPasswordValid) {
        return res.status(400).json({ message: 'Current password is incorrect' });
      }
    } else if (req.user.role === 'admin') {
      // Admin cannot change superadmin passwords
      if (user.role === 'superadmin') {
        return res.status(403).json({ message: 'You cannot change superadmin passwords' });
      }
      // Admin can only change passwords for users in their company
      if (user.company_id !== req.user.company_id) {
        return res.status(403).json({ message: 'You can only change passwords for users in your company' });
      }
    }
    // Superadmin can change any user's password without current password
    
    // Hash new password
    const hashedPassword = await bcrypt.hash(newPassword, 12);
    
    // Update password
    await user.update({ password: hashedPassword });
    
    res.json({ message: 'Password updated successfully' });
  } catch (error) {
    console.error('Error updating password:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
