const express = require('express');
const { body, validationResult } = require('express-validator');
const { RkapProgram, User, Company, ActivityLog, sequelize } = require('../models');
const { Op } = require('sequelize');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Get all programs with pagination and filters
router.get('/', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';
    
    const whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }
    
    // Add other filters
    if (req.query.year) {
      whereClause.year = req.query.year;
    }
    if (req.query.status) {
      whereClause.status = req.query.status;
    }
    if (req.query.type) {
      whereClause.type = req.query.type;
    }
    if (req.query.category) {
      whereClause.category = req.query.category;
    }
    if (req.query.company_id) {
      whereClause.company_id = req.query.company_id;
    }

    // Add search functionality (fixed)
    if (search && search.trim() !== '') {
      whereClause[Op.or] = [
        { program_name: { [Op.like]: `%${search.trim()}%` } },
        { category: { [Op.like]: `%${search.trim()}%` } },
        { sub_category: { [Op.like]: `%${search.trim()}%` } },
        { cost_center: { [Op.like]: `%${search.trim()}%` } },
        { beban: { [Op.like]: `%${search.trim()}%` } }
      ];
    }

    const { count, rows: programs } = await RkapProgram.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      limit,
      offset,
      order: [['created_at', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: programs,
      pagination: {
        currentPage: page,
        totalPages,
        totalItems: count,
        itemsPerPage: limit
      }
    });
  } catch (error) {
    console.error('Error fetching programs:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get available years from database
router.get('/years', authenticateToken, async (req, res) => {
  try {
    const whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }

    const years = await RkapProgram.findAll({
      where: whereClause,
      attributes: [
        [sequelize.fn('DISTINCT', sequelize.col('year')), 'year']
      ],
      order: [['year', 'DESC']],
      raw: true
    });

    const yearList = years.map(item => item.year).filter(year => year !== null);
    res.json(yearList);
  } catch (error) {
    console.error('Error fetching available years:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get program by ID
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const program = await RkapProgram.findByPk(req.params.id, {
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ]
    });

    if (!program) {
      return res.status(404).json({ message: 'Program not found' });
    }

    // Check if user has access to this program
    if (req.user.role !== 'superadmin' && program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied' });
    }

    res.json(program);
  } catch (error) {
    console.error('Error fetching program:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Create new program
router.post('/', authenticateToken, [
  body('program_name').notEmpty().withMessage('Program name is required'),
  body('type').isIn(['CAPEX', 'OPEX']).withMessage('Type must be CAPEX or OPEX'),
  body('proposed_amount').isNumeric().withMessage('Proposed amount must be a number'),
  body('year').isInt().withMessage('Year must be a number')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    // Determine status based on user role
    let programStatus = 'Pengajuan'; // Default for USER
    if (req.user.role === 'admin') {
      programStatus = 'Disetujui'; // ADMIN programs are auto-approved
    } else if (req.user.role === 'superadmin') {
      // SUPERADMIN should not be able to create programs
      return res.status(403).json({ message: 'Superadmin cannot create programs' });
    }

    const programData = {
      ...req.body,
      user_id: req.user.id,
      company_id: req.user.role === 'superadmin' ? req.body.company_id : req.user.company_id,
      status: programStatus
    };

    // Remove realized_amount for regular users (only admin can set it during creation)
    if (req.user.role === 'user' && programData.realized_amount) {
      delete programData.realized_amount;
    }

    const program = await RkapProgram.create(programData);

    // For USER: Create approval request for program creation
    if (req.user.role === 'user') {
      const { ApprovalRequest } = require('../models');
      await ApprovalRequest.create({
        rkap_program_id: program.id,
        requested_by_user_id: req.user.id,
        field_changed: 'program_creation',
        old_value: '',
        new_value: JSON.stringify({
          program_name: program.program_name,
          proposed_amount: program.proposed_amount,
          category: program.category,
          type: program.type,
          year: program.year
        }),
        status: 'pending'
      });
    }

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'CREATE_PROGRAM',
      details: {
        program_id: program.id,
        program_name: program.program_name
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.status(201).json(program);
  } catch (error) {
    console.error('Error creating program:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update program
router.put('/:id', authenticateToken, [
  body('program_name').optional().notEmpty().withMessage('Program name cannot be empty'),
  body('type').optional().isIn(['CAPEX', 'OPEX']).withMessage('Type must be CAPEX or OPEX'),
  body('proposed_amount').optional().isNumeric().withMessage('Proposed amount must be a number'),
  body('year').optional().isInt().withMessage('Year must be a number')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const program = await RkapProgram.findByPk(req.params.id);
    if (!program) {
      return res.status(404).json({ message: 'Program not found' });
    }

    // Check if user has access to this program
    if (req.user.role !== 'superadmin' && program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied' });
    }

    // Check if user can edit based on status
    if (req.user.role === 'user' && program.status === 'Selesai') {
      return res.status(403).json({ message: 'Cannot edit completed program' });
    }

    // Only ADMIN can set status to Selesai
    if (req.body.status === 'Selesai' && req.user.role !== 'admin') {
      return res.status(403).json({ message: 'Only admin can set status to Selesai' });
    }

    // USER cannot update approved_amount
    if (req.user.role === 'user' && req.body.approved_amount !== undefined) {
      return res.status(403).json({ message: 'Users cannot update approved amount' });
    }

    const oldData = { ...program.toJSON() };
    await program.update(req.body);

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'UPDATE_PROGRAM',
      details: {
        program_id: program.id,
        changes: req.body
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.json(program);
  } catch (error) {
    console.error('Error updating program:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete program
router.delete('/:id', authenticateToken, async (req, res) => {
  try {
    const program = await RkapProgram.findByPk(req.params.id);
    if (!program) {
      return res.status(404).json({ message: 'Program not found' });
    }

    // Check if user has access to this program
    if (req.user.role !== 'superadmin' && program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied' });
    }

    // SUPERADMIN cannot delete programs
    if (req.user.role === 'superadmin') {
      return res.status(403).json({ message: 'Superadmin cannot delete programs' });
    }

    // USER can only delete their own programs with Pengajuan status
    if (req.user.role === 'user' && (program.user_id !== req.user.id || program.status !== 'Pengajuan')) {
      return res.status(403).json({ message: 'Users can only delete their own programs with Pengajuan status' });
    }

    // ADMIN can delete any program in their company (except Selesai)
    if (req.user.role === 'admin' && program.status === 'Selesai') {
      return res.status(403).json({ message: 'Cannot delete completed programs' });
    }

    await program.destroy();

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'DELETE_PROGRAM',
      details: {
        program_id: program.id,
        program_name: program.program_name
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.json({ message: 'Program deleted successfully' });
  } catch (error) {
    console.error('Error deleting program:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get program statistics
router.get('/statistics/overview', authenticateToken, async (req, res) => {
  try {
    const whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }

    const stats = await RkapProgram.findAll({
      where: whereClause,
      attributes: [
        [sequelize.fn('COUNT', sequelize.col('id')), 'totalPrograms'],
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'totalProposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'totalApproved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'totalRealized']
      ],
      raw: true
    });

    res.json(stats[0]);
  } catch (error) {
    console.error('Error fetching program statistics:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get program statistics by status
router.get('/statistics/status', authenticateToken, async (req, res) => {
  try {
    const whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }

    const statusStats = await RkapProgram.findAll({
      where: whereClause,
      attributes: [
        'status',
        [sequelize.fn('COUNT', sequelize.col('id')), 'count']
      ],
      group: ['status'],
      raw: true
    });

    // Transform to object format
    const stats = {};
    statusStats.forEach(stat => {
      stats[stat.status] = parseInt(stat.count);
    });

    res.json(stats);
  } catch (error) {
    console.error('Error fetching program status statistics:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
