const express = require('express');
const { ActivityLog, SessionLog, ApiLog, User, Company, sequelize } = require('../models');
const { authenticateToken } = require('../middleware/auth');
const { Op } = require('sequelize');

const router = express.Router();

// Get activity logs with pagination and filters
router.get('/activity', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    let whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }
    
    // Add other filters
    if (req.query.action_type && req.query.action_type !== 'all') {
      whereClause.action_type = req.query.action_type;
    }
    if (req.query.user_id) {
      whereClause.user_id = req.query.user_id;
    }
    if (req.query.user) {
      whereClause.user_id = req.query.user;
    }
    if (req.query.dateFrom || req.query.date_from) {
      const dateFrom = req.query.dateFrom || req.query.date_from;
      if (dateFrom) {
        whereClause.timestamp = {
          [Op.gte]: new Date(dateFrom)
        };
      }
    }
    if (req.query.dateTo || req.query.date_to) {
      const dateTo = req.query.dateTo || req.query.date_to;
      if (dateTo) {
        whereClause.timestamp = {
          ...whereClause.timestamp,
          [Op.lte]: new Date(dateTo)
        };
      }
    }

    const { count, rows: logs } = await ActivityLog.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      limit,
      offset,
      order: [['timestamp', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: logs,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching activity logs:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get session logs with pagination and filters
router.get('/session', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    let whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }
    
    // Add other filters
    if (req.query.user_id) {
      whereClause.user_id = req.query.user_id;
    }
    if (req.query.user) {
      whereClause.user_id = req.query.user;
    }
    if (req.query.status) {
      whereClause.status = req.query.status;
    }
    if (req.query.dateFrom || req.query.date_from) {
      const dateFrom = req.query.dateFrom || req.query.date_from;
      if (dateFrom) {
        whereClause.login_time = {
          [Op.gte]: new Date(dateFrom)
        };
      }
    }
    if (req.query.dateTo || req.query.date_to) {
      const dateTo = req.query.dateTo || req.query.date_to;
      if (dateTo) {
        whereClause.login_time = {
          ...whereClause.login_time,
          [Op.lte]: new Date(dateTo)
        };
      }
    }

    const { count, rows: logs } = await SessionLog.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      limit,
      offset,
      order: [['login_time', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: logs,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching session logs:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get API logs with pagination and filters
router.get('/api', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    let whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }
    
    // Add other filters
    if (req.query.method) {
      whereClause.method = req.query.method;
    }
    if (req.query.endpoint) {
      whereClause.endpoint = {
        [Op.like]: `%${req.query.endpoint}%`
      };
    }
    if (req.query.status_code) {
      whereClause.status_code = req.query.status_code;
    }
    if (req.query.user_id) {
      whereClause.user_id = req.query.user_id;
    }
    if (req.query.user) {
      whereClause.user_id = req.query.user;
    }
    if (req.query.dateFrom || req.query.date_from) {
      const dateFrom = req.query.dateFrom || req.query.date_from;
      if (dateFrom) {
        whereClause.timestamp = {
          [Op.gte]: new Date(dateFrom)
        };
      }
    }
    if (req.query.dateTo || req.query.date_to) {
      const dateTo = req.query.dateTo || req.query.date_to;
      if (dateTo) {
        whereClause.timestamp = {
          ...whereClause.timestamp,
          [Op.lte]: new Date(dateTo)
        };
      }
    }

    const { count, rows: logs } = await ApiLog.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      limit,
      offset,
      order: [['timestamp', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: logs,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching API logs:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get log statistics
router.get('/statistics', authenticateToken, async (req, res) => {
  try {
    let whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }

    const [activityCount, sessionCount, apiCount] = await Promise.all([
      ActivityLog.count({ where: whereClause }),
      SessionLog.count({ where: whereClause }),
      ApiLog.count({ where: whereClause })
    ]);

    res.json({
      activityLogs: activityCount,
      sessionLogs: sessionCount,
      apiLogs: apiCount
    });
  } catch (error) {
    console.error('Error fetching log statistics:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Export logs
router.get('/export/:type', authenticateToken, async (req, res) => {
  try {
    const { type } = req.params;
    const { format = 'csv' } = req.query;
    
    let whereClause = {};
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id;
    }

    let logs = [];
    let filename = '';

    switch (type) {
      case 'activity':
        logs = await ActivityLog.findAll({
          where: whereClause,
          include: [
            { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
            { model: Company, as: 'company', attributes: ['id', 'name'] }
          ],
          order: [['timestamp', 'DESC']]
        });
        filename = 'activity_logs';
        break;
      case 'session':
        logs = await SessionLog.findAll({
          where: whereClause,
          include: [
            { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
            { model: Company, as: 'company', attributes: ['id', 'name'] }
          ],
          order: [['login_time', 'DESC']]
        });
        filename = 'session_logs';
        break;
      case 'api':
        logs = await ApiLog.findAll({
          where: whereClause,
          include: [
            { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
            { model: Company, as: 'company', attributes: ['id', 'name'] }
          ],
          order: [['timestamp', 'DESC']]
        });
        filename = 'api_logs';
        break;
      default:
        return res.status(400).json({ message: 'Invalid log type' });
    }

    if (format === 'csv') {
      // Convert to CSV format
      const csv = convertToCSV(logs);
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}.csv"`);
      res.send(csv);
    } else {
      res.json(logs);
    }
  } catch (error) {
    console.error('Error exporting logs:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Helper function to convert data to CSV
function convertToCSV(data) {
  if (data.length === 0) return '';
  
  const headers = Object.keys(data[0].toJSON());
  const csvHeaders = headers.join(',');
  
  const csvRows = data.map(row => {
    const values = headers.map(header => {
      const value = row[header];
      if (typeof value === 'object' && value !== null) {
        return JSON.stringify(value);
      }
      return value || '';
    });
    return values.join(',');
  });
  
  return [csvHeaders, ...csvRows].join('\n');
}

module.exports = router;
