const express = require('express');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const XLSX = require('xlsx');
const jsPDF = require('jspdf').jsPDF;
const { RkapProgram, User, Company } = require('../models');
const { authenticateToken: auth } = require('../middleware/auth');

const router = express.Router();

// Configure multer for file uploads
const upload = multer({
  dest: 'uploads/',
  fileFilter: (req, file, cb) => {
    if (file.mimetype === 'text/csv' || file.mimetype === 'application/vnd.ms-excel') {
      cb(null, true);
    } else {
      cb(new Error('Only CSV files are allowed'), false);
    }
  },
  limits: {
    fileSize: 5 * 1024 * 1024 // 5MB limit
  }
});

// Helper function to get programs based on user role
const getProgramsByRole = async (user) => {
  let whereClause = {};
  
  if (user.role === 'admin') {
    whereClause.company_id = user.company_id;
  } else if (user.role === 'user') {
    whereClause.user_id = user.id;
  }
  // superadmin can see all programs (no whereClause)
  
  return await RkapProgram.findAll({
    where: whereClause,
    include: [
      { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
      { model: Company, as: 'company', attributes: ['id', 'name'] }
    ],
    order: [['created_at', 'DESC']]
  });
};

// Export CSV
router.get('/export/csv', auth, async (req, res) => {
  try {
    const programs = await getProgramsByRole(req.user);
    
    // Create CSV headers
    const headers = [
      'ID', 'Program Name', 'Category', 'Sub Category', 'Cost Center',
      'Beban', 'Type', 'Proposed Amount', 'Approved Amount', 'Realized Amount',
      'Status', 'Notes', 'Year', 'User', 'Company', 'Created At', 'Updated At'
    ];
    
    // Convert to CSV format
    const csvData = programs.map(program => [
      program.id,
      program.program_name,
      program.category,
      program.sub_category,
      program.cost_center,
      program.beban,
      program.type,
      program.proposed_amount,
      program.approved_amount,
      program.realized_amount,
      program.status,
      program.notes,
      program.year,
      program.user?.name || '',
      program.company?.name || '',
      program.created_at,
      program.updated_at
    ]);
    
    const csvContent = [headers, ...csvData]
      .map(row => row.map(field => `"${field}"`).join(','))
      .join('\n');
    
    res.setHeader('Content-Type', 'text/csv');
    res.setHeader('Content-Disposition', `attachment; filename="programs_${new Date().toISOString().split('T')[0]}.csv"`);
    res.send(csvContent);
    
  } catch (error) {
    console.error('Export CSV error:', error);
    res.status(500).json({ error: 'Failed to export CSV' });
  }
});

// Export Excel
router.get('/export/excel', auth, async (req, res) => {
  try {
    const programs = await getProgramsByRole(req.user);
    
    // Prepare data for Excel
    const excelData = programs.map(program => ({
      'ID': program.id,
      'Program Name': program.program_name,
      'Category': program.category,
      'Sub Category': program.sub_category,
      'Cost Center': program.cost_center,
      'Beban': program.beban,
      'Type': program.type,
      'Proposed Amount': program.proposed_amount,
      'Approved Amount': program.approved_amount,
      'Realized Amount': program.realized_amount,
      'Status': program.status,
      'Notes': program.notes,
      'Year': program.year,
      'User': program.user?.name || '',
      'Company': program.company?.name || '',
      'Created At': program.created_at,
      'Updated At': program.updated_at
    }));
    
    // Create workbook and worksheet
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(excelData);
    
    // Set column widths
    const colWidths = [
      { wch: 8 },   // ID
      { wch: 30 },  // Program Name
      { wch: 15 },  // Category
      { wch: 20 },  // Sub Category
      { wch: 15 },  // Cost Center
      { wch: 15 },  // Beban
      { wch: 10 },  // Type
      { wch: 15 },  // Proposed Amount
      { wch: 15 },  // Approved Amount
      { wch: 15 },  // Realized Amount
      { wch: 12 },  // Status
      { wch: 30 },  // Notes
      { wch: 8 },   // Year
      { wch: 20 },  // User
      { wch: 20 },  // Company
      { wch: 20 },  // Created At
      { wch: 20 }   // Updated At
    ];
    ws['!cols'] = colWidths;
    
    XLSX.utils.book_append_sheet(wb, ws, 'Programs');
    
    // Generate buffer
    const buffer = XLSX.write(wb, { type: 'buffer', bookType: 'xlsx' });
    
    res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    res.setHeader('Content-Disposition', `attachment; filename="programs_${new Date().toISOString().split('T')[0]}.xlsx"`);
    res.send(buffer);
    
  } catch (error) {
    console.error('Export Excel error:', error);
    res.status(500).json({ error: 'Failed to export Excel' });
  }
});

// Export PDF
router.get('/export/pdf', auth, async (req, res) => {
  try {
    const programs = await getProgramsByRole(req.user);
    
    // Create PDF
    const doc = new jsPDF();
    
    // Add title
    doc.setFontSize(16);
    doc.text('RKAP Programs Report', 20, 20);
    
    // Add date
    doc.setFontSize(10);
    doc.text(`Generated: ${new Date().toLocaleDateString()}`, 20, 30);
    
    // Add user info
    doc.text(`User: ${req.user.name} (${req.user.role})`, 20, 40);
    
    // Add summary
    const totalPrograms = programs.length;
    const totalProposed = programs.reduce((sum, p) => sum + (p.proposed_amount || 0), 0);
    const totalApproved = programs.reduce((sum, p) => sum + (p.approved_amount || 0), 0);
    const totalRealized = programs.reduce((sum, p) => sum + (p.realized_amount || 0), 0);
    
    doc.text(`Total Programs: ${totalPrograms}`, 20, 50);
    doc.text(`Total Proposed: Rp ${totalProposed.toLocaleString()}`, 20, 60);
    doc.text(`Total Approved: Rp ${totalApproved.toLocaleString()}`, 20, 70);
    doc.text(`Total Realized: Rp ${totalRealized.toLocaleString()}`, 20, 80);
    
    // Add programs table
    let yPosition = 100;
    const pageHeight = doc.internal.pageSize.height;
    
    programs.forEach((program, index) => {
      // Check if we need a new page
      if (yPosition > pageHeight - 40) {
        doc.addPage();
        yPosition = 20;
      }
      
      doc.setFontSize(10);
      doc.text(`${index + 1}. ${program.program_name}`, 20, yPosition);
      doc.text(`   Category: ${program.category} | Type: ${program.type}`, 20, yPosition + 8);
      doc.text(`   Proposed: Rp ${(program.proposed_amount || 0).toLocaleString()} | Status: ${program.status}`, 20, yPosition + 16);
      
      yPosition += 30;
    });
    
    // Generate buffer
    const buffer = doc.output('arraybuffer');
    
    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Content-Disposition', `attachment; filename="programs_${new Date().toISOString().split('T')[0]}.pdf"`);
    res.send(Buffer.from(buffer));
    
  } catch (error) {
    console.error('Export PDF error:', error);
    res.status(500).json({ error: 'Failed to export PDF' });
  }
});

// Preview CSV Import
router.post('/import/csv/preview', auth, upload.single('file'), async (req, res) => {
  try {
    // Check if user can import
    if (req.user.role === 'user') {
      return res.status(403).json({ error: 'Users cannot import directly. Please request approval from admin.' });
    }
    
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }
    
    const results = [];
    const errors = [];
    const previewData = [];
    
    // Parse CSV file
    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', async () => {
        try {
          // Clean up uploaded file
          fs.unlinkSync(req.file.path);
          
          // Process each row for preview
          for (let i = 0; i < results.length; i++) {
            const row = results[i];
            const rowNumber = i + 2; // +2 because CSV starts from row 2 (after header)
            
            try {
              // Validate required fields
              if (!row['Program Name'] || !row['Category'] || !row['Type']) {
                errors.push(`Row ${rowNumber}: Missing required fields (Program Name, Category, Type)`);
                continue;
              }
              
              // Prepare preview data
              const previewRow = {
                rowNumber,
                program_name: row['Program Name'],
                category: row['Category'],
                sub_category: row['Sub Category'] || '',
                cost_center: row['Cost Center'] || '',
                beban: row['Beban'] || '',
                type: row['Type'],
                proposed_amount: parseFloat(row['Proposed Amount']) || 0,
                approved_amount: parseFloat(row['Approved Amount']) || 0,
                realized_amount: parseFloat(row['Realized Amount']) || 0,
                status: row['Status'] || 'draft',
                notes: row['Notes'] || '',
                year: parseInt(row['Year']) || new Date().getFullYear(),
                isValid: true
              };
              
              previewData.push(previewRow);
              
            } catch (error) {
              errors.push(`Row ${rowNumber}: ${error.message}`);
              previewData.push({
                rowNumber,
                program_name: row['Program Name'] || 'Invalid',
                category: row['Category'] || 'Invalid',
                sub_category: '',
                cost_center: '',
                beban: '',
                type: row['Type'] || 'Invalid',
                proposed_amount: 0,
                approved_amount: 0,
                realized_amount: 0,
                status: 'draft',
                notes: '',
                year: new Date().getFullYear(),
                isValid: false,
                error: error.message
              });
            }
          }
          
          res.json({
            message: 'Preview completed',
            totalRows: results.length,
            validRows: previewData.filter(row => row.isValid).length,
            invalidRows: errors.length,
            previewData: previewData.slice(0, 50), // Show first 50 rows for preview
            errors: errors.slice(0, 20), // Show first 20 errors
            hasMoreData: results.length > 50
          });
          
        } catch (error) {
          console.error('Preview processing error:', error);
          res.status(500).json({ error: 'Failed to process preview' });
        }
      })
      .on('error', (error) => {
        console.error('CSV parsing error:', error);
        fs.unlinkSync(req.file.path); // Clean up
        res.status(500).json({ error: 'Failed to parse CSV file' });
      });
      
  } catch (error) {
    console.error('Preview error:', error);
    res.status(500).json({ error: 'Failed to preview import' });
  }
});

// Import CSV
router.post('/import/csv', auth, upload.single('file'), async (req, res) => {
  try {
    // Check if user can import
    if (req.user.role === 'user') {
      return res.status(403).json({ error: 'Users cannot import directly. Please request approval from admin.' });
    }
    
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }
    
    const results = [];
    const errors = [];
    
    // Parse CSV file
    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', async () => {
        try {
          // Clean up uploaded file
          fs.unlinkSync(req.file.path);
          
          console.log(`Starting import of ${results.length} rows...`);
          let importedCount = 0;
          
          // Process each row
          for (let i = 0; i < results.length; i++) {
            const row = results[i];
            const rowNumber = i + 2; // +2 because CSV starts from row 2 (after header)
            
            try {
              // Validate required fields
              if (!row['Program Name'] || !row['Category'] || !row['Type']) {
                errors.push(`Row ${rowNumber}: Missing required fields (Program Name, Category, Type)`);
                continue;
              }
              
              // Prepare program data
              const programData = {
                program_name: row['Program Name'],
                category: row['Category'],
                sub_category: row['Sub Category'] || null,
                cost_center: row['Cost Center'] || null,
                beban: row['Beban'] || null,
                type: row['Type'],
                proposed_amount: parseFloat(row['Proposed Amount']) || 0,
                approved_amount: parseFloat(row['Approved Amount']) || 0,
                realized_amount: parseFloat(row['Realized Amount']) || 0,
                status: row['Status'] || 'Pengajuan', // Default to Pengajuan instead of draft
                notes: row['Notes'] || null,
                year: parseInt(row['Year']) || new Date().getFullYear(),
                user_id: req.user.id,
                company_id: req.user.company_id
              };
              
              // Create program
              await RkapProgram.create(programData);
              importedCount++;
              
              // Log progress every 10 rows
              if (importedCount % 10 === 0) {
                console.log(`Imported ${importedCount}/${results.length} rows...`);
              }
              
            } catch (error) {
              console.error(`Error importing row ${rowNumber}:`, error.message);
              errors.push(`Row ${rowNumber}: ${error.message}`);
            }
          }
          
          console.log(`Import completed. Total imported: ${importedCount}, Errors: ${errors.length}`);
          
          res.json({
            message: 'Import completed',
            imported: importedCount,
            totalRows: results.length,
            errors: errors.length,
            errorDetails: errors.slice(0, 50) // Limit error details to first 50
          });
          
        } catch (error) {
          console.error('Import processing error:', error);
          res.status(500).json({ error: 'Failed to process import' });
        }
      })
      .on('error', (error) => {
        console.error('CSV parsing error:', error);
        fs.unlinkSync(req.file.path); // Clean up
        res.status(500).json({ error: 'Failed to parse CSV file' });
      });
      
  } catch (error) {
    console.error('Import CSV error:', error);
    if (req.file) {
      fs.unlinkSync(req.file.path); // Clean up
    }
    res.status(500).json({ error: 'Failed to import CSV' });
  }
});

// Get CSV template
router.get('/template/csv', auth, (req, res) => {
  const template = [
    'Program Name,Category,Sub Category,Cost Center,Beban,Type,Proposed Amount,Approved Amount,Realized Amount,Status,Notes,Year',
    'Contoh Program Kerja,Operational,Maintenance,IT-001,Monthly,OPEX,10000000,9500000,0,draft,Program contoh untuk testing,2024',
    'Pengembangan Sistem,Capital,Software Development,IT-002,Annual,CAPEX,50000000,0,0,draft,Implementasi sistem baru,2024',
    'Pelatihan Karyawan,Operational,Training,HR-001,Quarterly,OPEX,15000000,0,0,draft,Program pelatihan karyawan,2024'
  ].join('\n');
  
  res.setHeader('Content-Type', 'text/csv');
  res.setHeader('Content-Disposition', 'attachment; filename="program_template.csv"');
  res.send(template);
});

module.exports = router;
