const express = require('express')
const { RkapProgram, Company, User, sequelize } = require('../models')
const { authenticateToken } = require('../middleware/auth')
const { Op } = require('sequelize')

const router = express.Router()

// Get available years from database
router.get('/years', authenticateToken, async (req, res) => {
  try {
    let whereClause = {}
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }
    
    // Get distinct years from programs
    const years = await RkapProgram.findAll({
      attributes: [
        [sequelize.fn('DISTINCT', sequelize.col('year')), 'year']
      ],
      where: whereClause,
      order: [['year', 'DESC']],
      raw: true
    })
    
    // Extract years and convert to numbers
    const availableYears = years
      .map(item => parseInt(item.year))
      .filter(year => !isNaN(year))
      .sort((a, b) => b - a) // Sort descending (newest first)
    
    
    res.json({
      years: availableYears,
      currentYear: new Date().getFullYear()
    })
  } catch (error) {
    console.error('Error fetching available years:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// Executive Summary
router.get('/executive-summary', authenticateToken, async (req, res) => {
  try {
    const { year } = req.query
    const yearFilter = year || new Date().getFullYear()

    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }


    // Get KPIs
    const kpis = await RkapProgram.findOne({
      attributes: [
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'totalProposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'totalApproved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'totalRealized'],
        [sequelize.fn('COUNT', sequelize.col('id')), 'totalPrograms']
      ],
      where: whereClause,
      raw: true
    })

    // Calculate absorption rate
    const absorptionRate = kpis.totalApproved > 0 
      ? (kpis.totalRealized / kpis.totalApproved) * 100 
      : 0

    const kpiData = {
      totalProposed: parseFloat(kpis.totalProposed) || 0,
      totalApproved: parseFloat(kpis.totalApproved) || 0,
      totalRealized: parseFloat(kpis.totalRealized) || 0,
      absorptionRate: parseFloat(absorptionRate) || 0,
      totalPrograms: parseInt(kpis.totalPrograms) || 0
    }

    // Budget per Company
    const budgetPerCompanyRaw = await RkapProgram.findAll({
      attributes: [
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'proposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized'],
        [sequelize.col('company.id'), 'companyId'],
        [sequelize.col('company.name'), 'companyName']
      ],
      include: [{
        model: Company,
        as: 'company',
        attributes: []
      }],
      where: whereClause,
      group: ['company.id'],
      raw: true
    })

    // Transform data to proper format
    const budgetPerCompany = budgetPerCompanyRaw.map(item => ({
      proposed: parseFloat(item.proposed) || 0,
      approved: parseFloat(item.approved) || 0,
      realized: parseFloat(item.realized) || 0,
      company: {
        id: item.companyId,
        name: item.companyName
      }
    }))

    // CAPEX vs OPEX Composition
    const capexOpexCompositionRaw = await RkapProgram.findAll({
      attributes: [
        'type',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'amount']
      ],
      where: whereClause,
      group: ['type'],
      raw: true
    })

    // Transform CAPEX vs OPEX data
    const capexOpexComposition = capexOpexCompositionRaw.map(item => ({
      type: item.type,
      amount: parseFloat(item.amount) || 0
    }))

    // Annual Trend - get data for current year and previous 2 years
    const annualTrendData = await RkapProgram.findAll({
      attributes: [
        'year',
        'type',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'total']
      ],
      where: {
        ...whereClause,
        year: {
          [Op.gte]: yearFilter - 2,
          [Op.lte]: yearFilter
        }
      },
      group: ['year', 'type'],
      order: [['year', 'ASC']],
      raw: true
    })

    // Transform data for chart
    const annualTrend = []
    for (let year = yearFilter - 2; year <= yearFilter; year++) {
      const yearData = { year }
      annualTrendData.forEach(item => {
        if (item.year === year) {
          yearData[item.type.toLowerCase()] = parseFloat(item.total) || 0
        }
      })
      annualTrend.push(yearData)
    }

    res.json({
      kpis: kpiData,
      budgetPerCompany,
      capexOpexComposition,
      annualTrend
    })
  } catch (error) {
    console.error('Error fetching executive summary:', error)
    console.error('Error details:', error.message)
    console.error('Stack trace:', error.stack)
    res.status(500).json({ message: 'Server error', error: error.message })
  }
})

// Company Analysis
router.get('/company-analysis', authenticateToken, async (req, res) => {
  try {
    const { year, company_id } = req.query
    const yearFilter = year || new Date().getFullYear()
    
    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    } else if (company_id) {
      whereClause.company_id = company_id
    }

    // Get KPIs for the company
    const kpis = await RkapProgram.findOne({
      attributes: [
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'totalProposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'totalApproved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'totalRealized'],
        [sequelize.fn('COUNT', sequelize.col('id')), 'totalPrograms']
      ],
      where: whereClause,
      raw: true
    })

    const absorptionRate = kpis.totalApproved > 0 
      ? (kpis.totalRealized / kpis.totalApproved) * 100 
      : 0

    const kpiData = {
      totalProposed: parseFloat(kpis.totalProposed) || 0,
      totalApproved: parseFloat(kpis.totalApproved) || 0,
      totalRealized: parseFloat(kpis.totalRealized) || 0,
      absorptionRate: parseFloat(absorptionRate) || 0,
      totalPrograms: parseInt(kpis.totalPrograms) || 0
    }

    // Budget by Category
    const budgetByCategoryRaw = await RkapProgram.findAll({
      attributes: [
        'category',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'proposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['category'],
      raw: true
    })

    // Transform budget by category data
    const budgetByCategory = budgetByCategoryRaw.map(item => ({
      category: item.category,
      proposed: parseFloat(item.proposed) || 0,
      approved: parseFloat(item.approved) || 0,
      realized: parseFloat(item.realized) || 0
    }))

    // CAPEX vs OPEX Composition for this company
    const capexOpexCompositionRaw = await RkapProgram.findAll({
      attributes: [
        'type',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'amount']
      ],
      where: whereClause,
      group: ['type'],
      raw: true
    })

    // Transform CAPEX vs OPEX data
    const capexOpexComposition = capexOpexCompositionRaw.map(item => ({
      type: item.type,
      amount: parseFloat(item.amount) || 0
    }))

    // Programs list
    const programs = await RkapProgram.findAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      order: [['created_at', 'DESC']],
      limit: 50
    })

    res.json({
      kpis: kpiData,
      budgetByCategory,
      capexOpexComposition,
      programs
    })
  } catch (error) {
    console.error('Error fetching company analysis:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// Category Analysis
router.get('/category-analysis', authenticateToken, async (req, res) => {
  try {
    const { year } = req.query
    const yearFilter = year || new Date().getFullYear()

    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }

    // Budget Distribution by Category
    const budgetDistribution = await RkapProgram.findAll({
      attributes: [
        'category',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized'],
        [sequelize.fn('COUNT', sequelize.col('id')), 'programs']
      ],
      where: whereClause,
      group: ['category'],
      order: [[sequelize.fn('SUM', sequelize.col('approved_amount')), 'DESC']],
      raw: true
    })

    // Get all sub-categories for drill-down (filtered by selectedCategory)
    const subCategories = await RkapProgram.findAll({
      attributes: [
        'category',
        'sub_category',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['category', 'sub_category'],
      raw: true
    })

    // Get all sub-categories without category filter (for left chart)
    const allSubCategories = await RkapProgram.findAll({
      attributes: [
        'sub_category',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['sub_category'],
      order: [[sequelize.fn('SUM', sequelize.col('approved_amount')), 'DESC']],
      raw: true
    })

    res.json({
      budgetDistribution,
      subCategories,
      allSubCategories
    })
  } catch (error) {
    console.error('Error fetching category analysis:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// CAPEX vs OPEX Analysis
router.get('/capex-opex-analysis', authenticateToken, async (req, res) => {
  try {
    const { year } = req.query
    const yearFilter = year || new Date().getFullYear()

    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }

    // Composition per Company
    const compositionPerCompanyRaw = await RkapProgram.findAll({
      attributes: [
        'type',
        [sequelize.fn('SUM', sequelize.col('RkapProgram.approved_amount')), 'amount'],
        [sequelize.col('company.id'), 'companyId'],
        [sequelize.col('company.name'), 'companyName']
      ],
      include: [{
        model: Company,
        as: 'company',
        attributes: ['id', 'name'],
        required: true
      }],
      where: whereClause,
      group: ['company.id', 'company.name', 'type'],
      raw: true
    })

    // Debug logging
    console.log('compositionPerCompanyRaw:', compositionPerCompanyRaw)

    // Transform data to proper format for chart
    const compositionPerCompany = compositionPerCompanyRaw.map(item => ({
      type: item.type,
      total: parseFloat(item.amount) || 0,
      company: {
        id: item.companyId,
        name: item.companyName || 'Unknown'
      }
    }))

    console.log('compositionPerCompany transformed:', compositionPerCompany)

    // Top CAPEX Programs
    const topCapexProgramsRaw = await RkapProgram.findAll({
      attributes: [
        'id',
        'program_name', 
        'approved_amount',
        [sequelize.col('company.id'), 'companyId'],
        [sequelize.col('company.name'), 'companyName']
      ],
      include: [{
        model: Company,
        as: 'company',
        attributes: ['id', 'name'],
        required: true
      }],
      where: { 
        ...whereClause,
        type: 'CAPEX'
      },
      order: [['approved_amount', 'DESC']],
      limit: 10,
      raw: true
    })

    // Debug logging
    console.log('topCapexProgramsRaw:', topCapexProgramsRaw)

    // Transform Top CAPEX Programs
    const topCapexPrograms = topCapexProgramsRaw.map(item => ({
      id: item.id,
      program_name: item.program_name,
      approved_amount: parseFloat(item.approved_amount) || 0,
      company: {
        id: item.companyId,
        name: item.companyName || 'Unknown'
      }
    }))

    console.log('topCapexPrograms transformed:', topCapexPrograms)

    // Top OPEX Programs
    const topOpexProgramsRaw = await RkapProgram.findAll({
      attributes: [
        'id',
        'program_name', 
        'approved_amount',
        [sequelize.col('company.id'), 'companyId'],
        [sequelize.col('company.name'), 'companyName']
      ],
      include: [{
        model: Company,
        as: 'company',
        attributes: ['id', 'name'],
        required: true
      }],
      where: { 
        ...whereClause,
        type: 'OPEX'
      },
      order: [['approved_amount', 'DESC']],
      limit: 10,
      raw: true
    })

    // Debug logging
    console.log('topOpexProgramsRaw:', topOpexProgramsRaw)

    // Transform Top OPEX Programs
    const topOpexPrograms = topOpexProgramsRaw.map(item => ({
      id: item.id,
      program_name: item.program_name,
      approved_amount: parseFloat(item.approved_amount) || 0,
      company: {
        id: item.companyId,
        name: item.companyName || 'Unknown'
      }
    }))

    console.log('topOpexPrograms transformed:', topOpexPrograms)

    res.json({
      compositionPerCompany,
      topCapexPrograms,
      topOpexPrograms
    })
  } catch (error) {
    console.error('Error fetching CAPEX/OPEX analysis:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// Cost Center Analysis
router.get('/cost-center-analysis', authenticateToken, async (req, res) => {
  try {
    const { year } = req.query
    const yearFilter = year || new Date().getFullYear()

    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }

    // Get KPIs
    const kpis = await RkapProgram.findOne({
      attributes: [
        [sequelize.fn('COUNT', sequelize.fn('DISTINCT', sequelize.col('cost_center'))), 'totalCostCenters'],
        [sequelize.fn('COUNT', sequelize.fn('DISTINCT', sequelize.col('beban'))), 'totalBebanTypes'],
        [sequelize.fn('AVG', sequelize.literal('(realized_amount / approved_amount) * 100')), 'avgRealizationRate']
      ],
      where: whereClause,
      raw: true
    })

    // Budget Distribution by Cost Center
    const budgetByCostCenterRaw = await RkapProgram.findAll({
      attributes: [
        'cost_center',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'amount']
      ],
      where: whereClause,
      group: ['cost_center'],
      order: [[sequelize.fn('SUM', sequelize.col('approved_amount')), 'DESC']],
      raw: true
    })

    // Transform and filter data
    const budgetByCostCenter = budgetByCostCenterRaw
      .filter(item => item.cost_center && item.cost_center.trim() !== '')
      .map(item => ({
        cost_center: item.cost_center,
        amount: parseFloat(item.amount) || 0
      }))

    console.log('budgetByCostCenterRaw:', budgetByCostCenterRaw)
    console.log('budgetByCostCenter transformed:', budgetByCostCenter)

    // Realization Rate by Cost Center
    const realizationByCostCenterRaw = await RkapProgram.findAll({
      attributes: [
        'cost_center',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['cost_center'],
      raw: true
    })

    // Transform and calculate realization rate
    const realizationByCostCenter = realizationByCostCenterRaw
      .filter(item => item.cost_center && item.cost_center.trim() !== '')
      .map(item => {
        const approved = parseFloat(item.approved) || 0
        const realized = parseFloat(item.realized) || 0
        const realizationRate = approved > 0 ? (realized / approved) * 100 : 0
        
        return {
          cost_center: item.cost_center,
          approved: approved,
          realized: realized,
          realizationRate: Math.round(realizationRate * 100) / 100 // Round to 2 decimal places
        }
      })
      .sort((a, b) => b.realizationRate - a.realizationRate)

    console.log('realizationByCostCenterRaw:', realizationByCostCenterRaw)
    console.log('realizationByCostCenter transformed:', realizationByCostCenter)

    // Budget Distribution by Beban Type
    const budgetByBebanRaw = await RkapProgram.findAll({
      attributes: [
        'beban',
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'amount']
      ],
      where: whereClause,
      group: ['beban'],
      order: [[sequelize.fn('SUM', sequelize.col('approved_amount')), 'DESC']],
      raw: true
    })

    // Transform and filter data
    const budgetByBeban = budgetByBebanRaw
      .filter(item => item.beban && item.beban.trim() !== '')
      .map(item => ({
        beban: item.beban,
        amount: parseFloat(item.amount) || 0
      }))

    console.log('budgetByBebanRaw:', budgetByBebanRaw)
    console.log('budgetByBeban transformed:', budgetByBeban)

    // Program Count by Beban Type
    const programCountByBebanRaw = await RkapProgram.findAll({
      attributes: [
        'beban',
        [sequelize.fn('COUNT', sequelize.col('id')), 'count']
      ],
      where: whereClause,
      group: ['beban'],
      order: [[sequelize.fn('COUNT', sequelize.col('id')), 'DESC']],
      raw: true
    })

    // Transform and filter data
    const programCountByBeban = programCountByBebanRaw
      .filter(item => item.beban && item.beban.trim() !== '')
      .map(item => ({
        beban: item.beban,
        count: parseInt(item.count) || 0
      }))

    console.log('programCountByBebanRaw:', programCountByBebanRaw)
    console.log('programCountByBeban transformed:', programCountByBeban)

    res.json({
      kpis,
      budgetByCostCenter,
      realizationByCostCenter,
      budgetByBeban,
      programCountByBeban
    })
  } catch (error) {
    console.error('Error fetching cost center analysis:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// Get companies for dropdown
router.get('/companies', authenticateToken, async (req, res) => {
  try {
    let whereClause = {}
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.id = req.user.company_id
    }

    const companies = await Company.findAll({
      where: whereClause,
      attributes: ['id', 'name'],
      order: [['name', 'ASC']]
    })

    res.json(companies)
  } catch (error) {
    console.error('Error fetching companies:', error)
    res.status(500).json({ message: 'Server error' })
  }
})

// Get categories for dropdown
router.get('/categories', authenticateToken, async (req, res) => {
  try {
    const { year } = req.query
    const yearFilter = year || new Date().getFullYear()

    let whereClause = { year: yearFilter }
    
    // Filter by company if user is not superadmin
    if (req.user.role !== 'superadmin') {
      whereClause.company_id = req.user.company_id
    }

    const categories = await RkapProgram.findAll({
      attributes: [
        [sequelize.fn('DISTINCT', sequelize.col('category')), 'category']
      ],
      where: {
        ...whereClause,
        category: { [Op.ne]: null }
      },
      order: [['category', 'ASC']],
      raw: true
    })

    const categoryList = categories.map(cat => cat.category).filter(Boolean)

    res.json(categoryList)
  } catch (error) {
    console.error('Error fetching categories:', error)
    console.error('Error details:', error.message)
    console.error('Stack trace:', error.stack)
    res.status(500).json({ message: 'Server error', error: error.message })
  }
})

module.exports = router