const express = require('express');
const { body, validationResult } = require('express-validator');
const { Company, User, ActivityLog } = require('../models');
const { authenticateToken, authorizeRoles } = require('../middleware/auth');

const router = express.Router();

// Get all companies (only superadmin can see all)
router.get('/', authenticateToken, async (req, res) => {
  try {
    let companies;
    
    if (req.user.role === 'superadmin') {
      // Superadmin can see all companies
      companies = await Company.findAll({
        attributes: ['id', 'name', 'created_at'],
        order: [['name', 'ASC']]
      });
    } else {
      // Admin and User can only see their own company
      companies = await Company.findAll({
        where: { id: req.user.company_id },
        attributes: ['id', 'name', 'created_at'],
        order: [['name', 'ASC']]
      });
    }

    res.json(companies);
  } catch (error) {
    console.error('Error fetching companies:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get company by ID
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const company = await Company.findByPk(req.params.id, {
      attributes: ['id', 'name', 'created_at']
    });

    if (!company) {
      return res.status(404).json({ message: 'Company not found' });
    }

    // Check if user has access to this company
    if (req.user.role !== 'superadmin' && company.id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied' });
    }

    res.json(company);
  } catch (error) {
    console.error('Error fetching company:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Create new company (only superadmin)
router.post('/', authenticateToken, authorizeRoles(['superadmin']), [
  body('name').notEmpty().withMessage('Company name is required')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const company = await Company.create(req.body);

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'CREATE_COMPANY',
      details: {
        company_id: company.id,
        company_name: company.name
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.status(201).json(company);
  } catch (error) {
    console.error('Error creating company:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update company (only superadmin)
router.put('/:id', authenticateToken, authorizeRoles(['superadmin']), [
  body('name').optional().notEmpty().withMessage('Company name cannot be empty')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const company = await Company.findByPk(req.params.id);
    if (!company) {
      return res.status(404).json({ message: 'Company not found' });
    }

    const oldData = { ...company.toJSON() };
    await company.update(req.body);

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'UPDATE_COMPANY',
      details: {
        company_id: company.id,
        changes: req.body
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.json(company);
  } catch (error) {
    console.error('Error updating company:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete company (only superadmin)
router.delete('/:id', authenticateToken, authorizeRoles(['superadmin']), async (req, res) => {
  try {
    const company = await Company.findByPk(req.params.id);
    if (!company) {
      return res.status(404).json({ message: 'Company not found' });
    }

    await company.destroy();

    // Log activity
    await ActivityLog.create({
      user_id: req.user.id,
      company_id: req.user.company_id,
      action_type: 'DELETE_COMPANY',
      details: {
        company_id: company.id,
        company_name: company.name
      },
      ip_address: req.ip,
      user_agent: req.get('User-Agent')
    });

    res.json({ message: 'Company deleted successfully' });
  } catch (error) {
    console.error('Error deleting company:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
