const express = require('express');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { body, validationResult } = require('express-validator');
const { User, Company } = require('../models');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Password complexity validation
const passwordComplexity = (password) => {
  const minLength = 8;
  const hasUpperCase = /[A-Z]/.test(password);
  const hasLowerCase = /[a-z]/.test(password);
  const hasNumbers = /\d/.test(password);
  const hasSpecialChar = /[!@#$%^&*(),.?":{}|<>]/.test(password);
  
  return {
    isValid: password.length >= minLength && hasUpperCase && hasLowerCase && hasNumbers && hasSpecialChar,
    minLength: password.length >= minLength,
    hasUpperCase,
    hasLowerCase,
    hasNumbers,
    hasSpecialChar
  };
};

// Register new user (only for 'user' role)
router.post('/register', [
  body('name').notEmpty().withMessage('Name is required'),
  body('email').isEmail().withMessage('Valid email is required'),
  body('password').custom((value) => {
    const complexity = passwordComplexity(value);
    if (!complexity.isValid) {
      throw new Error('Password must be at least 8 characters with uppercase, lowercase, numbers, and special characters');
    }
    return true;
  }),
  body('company_id').isInt().withMessage('Company ID is required and must be an integer')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, email, password, company_id } = req.body;

    // Check if user already exists
    const existingUser = await User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).json({ message: 'User already exists' });
    }

    // Check if company exists
    const company = await Company.findByPk(company_id);
    if (!company) {
      return res.status(400).json({ message: 'Company not found' });
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 12);

    // Create user (only 'user' role allowed)
    const user = await User.create({
      name,
      email,
      password: hashedPassword,
      role: 'user', // Fixed to 'user' role
      company_id,
      status: 'pending' // New users need admin approval
    });

    // Don't generate token for pending users
    res.status(201).json({
      message: 'User created successfully. Your account is pending approval from your company admin.',
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
        role: user.role,
        company_id: user.company_id,
        status: user.status,
        company: company
      }
    });
  } catch (error) {
    console.error('Registration error:', error);
    res.status(500).json({ message: 'Server error during registration' });
  }
});

// Get companies for registration
router.get('/companies', async (req, res) => {
  try {
    const companies = await Company.findAll({
      attributes: ['id', 'name'],
      order: [['name', 'ASC']]
    });
    res.json(companies);
  } catch (error) {
    console.error('Error fetching companies:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Login user
router.post('/login', [
  body('email').isEmail().withMessage('Valid email is required'),
  body('password').notEmpty().withMessage('Password is required')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { email, password } = req.body;

    // Find user
    const user = await User.findOne({ 
      where: { email },
      include: [{ model: require('../models').Company, as: 'company' }]
    });

    if (!user) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    // Check password
    const isPasswordValid = await bcrypt.compare(password, user.password);
    if (!isPasswordValid) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    // Check user approval status
    if (user.status === 'pending') {
      return res.status(403).json({ 
        message: 'Your account is pending approval. Please contact your company admin to approve your account.' 
      });
    }

    if (user.status === 'rejected') {
      return res.status(403).json({ 
        message: 'Your account has been rejected. Please contact your company admin for more information.' 
      });
    }

    // Generate JWT token
    const token = jwt.sign(
      { userId: user.id, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || '24h' }
    );

    res.json({
      message: 'Login successful',
      token,
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
        role: user.role,
        company_id: user.company_id,
        company: user.company
      }
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ message: 'Server error during login' });
  }
});

// Get current user profile
router.get('/profile', authenticateToken, async (req, res) => {
  try {
    const user = await User.findByPk(req.user.id, {
      include: [{ model: require('../models').Company, as: 'company' }],
      attributes: { exclude: ['password'] }
    });

    res.json({ user });
  } catch (error) {
    console.error('Profile error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update user profile
router.put('/profile', authenticateToken, [
  body('name').optional().notEmpty().withMessage('Name cannot be empty'),
  body('email').optional().isEmail().withMessage('Valid email is required'),
  body('ai_provider').optional().isIn(['gemini', 'openai', 'deepseek']).withMessage('Invalid AI provider'),
  body('ai_api_key').optional().isString().withMessage('AI API key must be a string')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, email, ai_provider, ai_api_key } = req.body;
    const updateData = {};

    if (name) updateData.name = name;
    if (email) {
      // Check if email is already taken by another user
      const existingUser = await User.findOne({ 
        where: { email, id: { [require('sequelize').Op.ne]: req.user.id } }
      });
      if (existingUser) {
        return res.status(400).json({ message: 'Email already in use' });
      }
      updateData.email = email;
    }
    if (ai_provider) updateData.ai_provider = ai_provider;
    if (ai_api_key !== undefined) updateData.ai_api_key = ai_api_key;

    await User.update(updateData, { where: { id: req.user.id } });

    const updatedUser = await User.findByPk(req.user.id, {
      include: [{ model: require('../models').Company, as: 'company' }],
      attributes: { exclude: ['password'] }
    });

    res.json({ 
      message: 'Profile updated successfully',
      user: updatedUser
    });
  } catch (error) {
    console.error('Profile update error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Change password
router.put('/password', authenticateToken, [
  body('currentPassword').notEmpty().withMessage('Current password is required'),
  body('newPassword').custom((value) => {
    const complexity = passwordComplexity(value);
    if (!complexity.isValid) {
      throw new Error('Password must be at least 8 characters with uppercase, lowercase, numbers, and special characters');
    }
    return true;
  })
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { currentPassword, newPassword } = req.body;

    // Get current user with password
    const user = await User.findByPk(req.user.id);
    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }

    // Verify current password
    const isCurrentPasswordValid = await bcrypt.compare(currentPassword, user.password);
    if (!isCurrentPasswordValid) {
      return res.status(400).json({ message: 'Current password is incorrect' });
    }

    // Hash new password
    const hashedNewPassword = await bcrypt.hash(newPassword, 12);

    // Update password
    await User.update({ password: hashedNewPassword }, { where: { id: req.user.id } });

    res.json({ message: 'Password changed successfully' });
  } catch (error) {
    console.error('Password change error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get pending users for admin approval
router.get('/pending-users', authenticateToken, async (req, res) => {
  try {
    // Only admin and superadmin can access this
    if (req.user.role === 'user') {
      return res.status(403).json({ message: 'Access denied' });
    }

    let whereClause = { status: 'pending' };
    
    // Admin can only see users from their company
    if (req.user.role === 'admin') {
      whereClause.company_id = req.user.company_id;
    }

    const pendingUsers = await User.findAll({
      where: whereClause,
      include: [{ model: require('../models').Company, as: 'company' }],
      attributes: { exclude: ['password'] },
      order: [['created_at', 'ASC']]
    });

    res.json({ users: pendingUsers });
  } catch (error) {
    console.error('Error fetching pending users:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Approve or reject user
router.put('/user/:id/approve', authenticateToken, [
  body('status').isIn(['approved', 'rejected']).withMessage('Status must be approved or rejected'),
  body('notes').optional().isString().withMessage('Notes must be a string')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    // Only admin and superadmin can approve users
    if (req.user.role === 'user') {
      return res.status(403).json({ message: 'Access denied' });
    }

    const { status, notes } = req.body;
    const userId = req.params.id;

    // Get user to approve
    const userToApprove = await User.findByPk(userId, {
      include: [{ model: require('../models').Company, as: 'company' }]
    });

    if (!userToApprove) {
      return res.status(404).json({ message: 'User not found' });
    }

    // Admin can only approve users from their company
    if (req.user.role === 'admin' && userToApprove.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'You can only approve users from your company' });
    }

    // Update user status
    await User.update({ status }, { where: { id: userId } });

    // Get updated user
    const updatedUser = await User.findByPk(userId, {
      include: [{ model: require('../models').Company, as: 'company' }],
      attributes: { exclude: ['password'] }
    });

    res.json({ 
      message: `User ${status} successfully`,
      user: updatedUser
    });
  } catch (error) {
    console.error('Error approving user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
