const express = require('express');
const { ApprovalRequest, RkapProgram, User, Company } = require('../models');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Get approval requests for admin/superadmin
router.get('/', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    let whereClause = {};
    
    // Filter by company if user is admin (not superadmin)
    let includeOptions = [
      {
        model: RkapProgram,
        as: 'rkapProgram',
        include: [
          { model: Company, as: 'company', attributes: ['id', 'name'] }
        ]
      },
      {
        model: User,
        as: 'requested_by',
        attributes: ['id', 'name', 'email']
      }
    ];
    
    if (req.user.role === 'admin') {
      includeOptions[0].where = { company_id: req.user.company_id };
    }
    
    const { count, rows: requests } = await ApprovalRequest.findAndCountAll({
      where: whereClause,
      include: includeOptions,
      limit,
      offset,
      order: [['created_at', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    // Debug logging for response
    console.log('🔍 Debug - Backend response:');
    console.log('🔍 Debug - requests count:', requests.length);
    console.log('🔍 Debug - first request:', requests[0] ? {
      id: requests[0].id,
      hasRkapProgram: !!requests[0].rkapProgram,
      hasRequestedBy: !!requests[0].requested_by,
      rkapProgramName: requests[0].rkapProgram?.program_name,
      requestedByName: requests[0].requested_by?.name
    } : 'No requests');

    res.json({
      data: requests,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching approval requests:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get approval requests for the current user (for users to see their own requests)
router.get('/my-requests', authenticateToken, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    // Only users can access this endpoint
    if (req.user.role !== 'user') {
      return res.status(403).json({ message: 'Only users can access this endpoint' });
    }
    
    const { count, rows: requests } = await ApprovalRequest.findAndCountAll({
      where: {
        requested_by_user_id: req.user.id
      },
      include: [
        {
          model: RkapProgram,
          as: 'rkapProgram',
          include: [
            { model: Company, as: 'company', attributes: ['id', 'name'] }
          ]
        },
        {
          model: User,
          as: 'requested_by',
          attributes: ['id', 'name', 'email']
        }
      ],
      limit,
      offset,
      order: [['created_at', 'DESC']]
    });

    const totalPages = Math.ceil(count / limit);

    res.json({
      data: requests,
      totalPages,
      currentPage: page,
      totalCount: count
    });
  } catch (error) {
    console.error('Error fetching user approval requests:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Submit approval request (alias for POST /)
router.post('/submit', authenticateToken, async (req, res) => {
  // Forward to the main POST route
  req.url = '/';
  return router.handle(req, res);
});

// Create approval request (for users when updating realization)
router.post('/', authenticateToken, async (req, res) => {
  try {
    const { program_id, field_changed, old_value, new_value } = req.body;
    
    // Only users can create approval requests for realization updates
    if (req.user.role !== 'user') {
      return res.status(403).json({ message: 'Only users can create approval requests' });
    }
    
    // Verify the program belongs to the user
    const program = await RkapProgram.findOne({
      where: { id: program_id, user_id: req.user.id }
    });
    
    if (!program) {
      return res.status(404).json({ message: 'Program not found or access denied' });
    }
    
    const approvalRequest = await ApprovalRequest.create({
      rkap_program_id: program_id,
      requested_by_user_id: req.user.id,
      field_changed,
      old_value,
      new_value,
      status: 'pending'
    });
    
    res.status(201).json(approvalRequest);
  } catch (error) {
    console.error('Error creating approval request:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Approve request
router.post('/:id/approve', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { admin_notes } = req.body;
    
    // Only admin and superadmin can approve
    if (!['admin', 'superadmin'].includes(req.user.role)) {
      return res.status(403).json({ message: 'Only admin and superadmin can approve requests' });
    }
    
    const request = await ApprovalRequest.findByPk(id, {
      include: [
        { model: RkapProgram, as: 'rkapProgram' }
      ]
    });
    
    if (!request) {
      return res.status(404).json({ message: 'Approval request not found' });
    }
    
    // Check company access for admin
    if (req.user.role === 'admin' && request.rkap_program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied to this company data' });
    }
    
    // Update approval request status
    await request.update({
      status: 'approved',
      admin_notes: admin_notes || null
    });
    
    // Apply the change to the program
    if (request.field_changed === 'program_creation') {
      // For program creation, update the status to 'Disetujui'
      await request.rkap_program.update({ status: 'Disetujui' });
    } else {
      // For other field changes, update the specific field
      const updateData = {};
      updateData[request.field_changed] = request.new_value;
      await request.rkap_program.update(updateData);
    }
    
    res.json({ message: 'Request approved successfully' });
  } catch (error) {
    console.error('Error approving request:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Reject request
router.post('/:id/reject', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { admin_notes } = req.body;
    
    // Only admin and superadmin can reject
    if (!['admin', 'superadmin'].includes(req.user.role)) {
      return res.status(403).json({ message: 'Only admin and superadmin can reject requests' });
    }
    
    const request = await ApprovalRequest.findByPk(id, {
      include: [
        { model: RkapProgram, as: 'rkapProgram' }
      ]
    });
    
    if (!request) {
      return res.status(404).json({ message: 'Approval request not found' });
    }
    
    // Check company access for admin
    if (req.user.role === 'admin' && request.rkap_program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied to this company data' });
    }
    
    // Update approval request status
    await request.update({
      status: 'rejected',
      admin_notes: admin_notes || null
    });
    
    res.json({ message: 'Request rejected successfully' });
  } catch (error) {
    console.error('Error rejecting request:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get approval request by ID
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    const request = await ApprovalRequest.findByPk(id, {
      include: [
        {
          model: RkapProgram,
          as: 'rkapProgram',
          include: [
            { model: Company, as: 'company', attributes: ['id', 'name'] }
          ]
        },
        {
          model: User,
          as: 'requested_by',
          attributes: ['id', 'name', 'email']
        }
      ]
    });
    
    if (!request) {
      return res.status(404).json({ message: 'Approval request not found' });
    }
    
    // Check access permissions
    if (req.user.role === 'admin' && request.rkap_program.company_id !== req.user.company_id) {
      return res.status(403).json({ message: 'Access denied to this company data' });
    }
    
    if (req.user.role === 'user' && request.requested_by_user_id !== req.user.id) {
      return res.status(403).json({ message: 'Access denied to this request' });
    }
    
    res.json(request);
  } catch (error) {
    console.error('Error fetching approval request:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
