const express = require('express')
const { User, RkapProgram, Company, AiResponse, sequelize } = require('../models')
const { authenticateToken } = require('../middleware/auth')
const { Op } = require('sequelize')
const { GoogleGenerativeAI } = require('@google/generative-ai')
const OpenAI = require('openai')

const router = express.Router()

// Helper function to get AI client based on provider
const getAIClient = (provider, apiKey) => {
  switch (provider) {
    case 'gemini':
      return new GoogleGenerativeAI(apiKey)
    case 'openai':
      return new OpenAI({ apiKey })
    case 'deepseek':
      return new OpenAI({ 
        apiKey,
        baseURL: 'https://api.deepseek.com'
      })
    default:
      throw new Error('Unsupported AI provider')
  }
}

// Helper function to format data for AI analysis
const formatDataForAI = (data) => {
  const summary = {
    totalPrograms: data.totalPrograms || 0,
    totalProposed: data.totalProposed || 0,
    totalApproved: data.totalApproved || 0,
    totalRealized: data.totalRealized || 0,
    absorptionRate: data.absorptionRate || 0,
    companies: data.companies || [],
    categories: data.categories || [],
    capexOpex: data.capexOpex || []
  }
  
  return JSON.stringify(summary, null, 2)
}

// Test AI connection
router.post('/test-connection', authenticateToken, async (req, res) => {
  try {
    const { provider, apiKey } = req.body
    
    if (!provider || !apiKey) {
      return res.status(400).json({ 
        success: false, 
        message: 'Provider and API key are required' 
      })
    }

    const client = getAIClient(provider, apiKey)
    
    // Test connection with a simple query
    let testResponse
    if (provider === 'gemini') {
      const model = client.getGenerativeModel({ model: 'gemini-2.0-flash-exp' })
      const result = await model.generateContent('Hello, test connection')
      testResponse = result.response.text()
    } else if (provider === 'openai' || provider === 'deepseek') {
      const completion = await client.chat.completions.create({
        model: provider === 'deepseek' ? 'deepseek-chat' : 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: 'Hello, test connection' }],
        max_tokens: 50
      })
      testResponse = completion.choices[0].message.content
    }

    // Update user's AI settings
    await User.update(
      { ai_provider: provider, ai_api_key: apiKey },
      { where: { id: req.user.id } }
    )

    res.json({
      success: true,
      message: 'AI connection successful',
      provider,
      testResponse: testResponse.substring(0, 100) + '...'
    })

  } catch (error) {
    console.error('AI connection test error:', error)
    res.status(400).json({
      success: false,
      message: 'AI connection failed: ' + error.message
    })
  }
})

// Analyze query with AI
router.post('/analyze', authenticateToken, async (req, res) => {
  try {
    console.log('🔍 Debug - AI analyze endpoint hit')
    console.log('🔍 Debug - Request body:', req.body)
    console.log('🔍 Debug - User from token:', req.user)
    
    const { query, year, companyId, companyIds, yearStart, yearEnd } = req.body
    const user = await User.findByPk(req.user.id, {
      include: [{ model: Company, as: 'company' }]
    })
    
    console.log('🔍 Debug - User found:', {
      id: user?.id,
      name: user?.name,
      ai_provider: user?.ai_provider,
      has_api_key: !!user?.ai_api_key,
      company_id: user?.company_id
    })
    
    if (!user.ai_provider || !user.ai_api_key) {
      return res.status(400).json({
        success: false,
        message: 'AI provider not configured. Please set up AI settings first.'
      })
    }

    // Only check company_id for non-superadmin users
    if (user.role !== 'superadmin' && !user.company_id) {
      return res.status(400).json({
        success: false,
        message: 'User company not found. Please contact administrator.'
      })
    }

    if (!query || !query.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Query is required'
      })
    }

    // Get user's data for analysis
    let whereClause = {}
    
    // Apply year filter
    if (year) {
      whereClause.year = parseInt(year)
    } else if (yearStart && yearEnd) {
      whereClause.year = {
        [Op.between]: [parseInt(yearStart), parseInt(yearEnd)]
      }
    }
    
    // Apply company filter
    if (user.role !== 'superadmin') {
      whereClause.company_id = user.company_id
    } else {
      // For superadmin, handle different company filter options
      if (companyIds && Array.isArray(companyIds) && companyIds.length > 0) {
        whereClause.company_id = {
          [Op.in]: companyIds.map(id => parseInt(id))
        }
      } else if (companyId) {
        whereClause.company_id = parseInt(companyId)
      }
      // If no company filter specified, superadmin gets all companies
    }
    
    console.log('🔍 Debug - Where clause:', whereClause)

    // Fetch relevant data
    const programs = await RkapProgram.findAll({
      where: whereClause,
      include: [
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ],
      limit: 100 // Limit for performance
    })

    const summary = await RkapProgram.findOne({
      attributes: [
        [sequelize.fn('COUNT', sequelize.col('id')), 'totalPrograms'],
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'totalProposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'totalApproved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'totalRealized']
      ],
      where: whereClause,
      raw: true
    })

    const categoryData = await RkapProgram.findAll({
      attributes: [
        'category',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'proposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['category'],
      raw: true
    })

    const companyData = await RkapProgram.findAll({
      attributes: [
        'company_id',
        [sequelize.fn('SUM', sequelize.col('proposed_amount')), 'proposed'],
        [sequelize.fn('SUM', sequelize.col('approved_amount')), 'approved'],
        [sequelize.fn('SUM', sequelize.col('realized_amount')), 'realized']
      ],
      where: whereClause,
      group: ['company_id'],
      include: [
        { model: Company, as: 'company', attributes: ['name'] }
      ],
      raw: true
    })

    // Format data for AI
    const dataForAI = {
      summary: {
        totalPrograms: parseInt(summary.totalPrograms) || 0,
        totalProposed: parseFloat(summary.totalProposed) || 0,
        totalApproved: parseFloat(summary.totalApproved) || 0,
        totalRealized: parseFloat(summary.totalRealized) || 0,
        absorptionRate: summary.totalApproved > 0 ? 
          ((parseFloat(summary.totalRealized) / parseFloat(summary.totalApproved)) * 100).toFixed(2) : 0
      },
      categories: categoryData.map(item => ({
        category: item.category,
        proposed: parseFloat(item.proposed) || 0,
        approved: parseFloat(item.approved) || 0,
        realized: parseFloat(item.realized) || 0
      })),
      companies: companyData.map(item => ({
        company: item['company.name'],
        proposed: parseFloat(item.proposed) || 0,
        approved: parseFloat(item.approved) || 0,
        realized: parseFloat(item.realized) || 0
      })),
      recentPrograms: programs.slice(0, 10).map(program => ({
        name: program.name,
        category: program.category,
        proposed: parseFloat(program.proposed_amount) || 0,
        approved: parseFloat(program.approved_amount) || 0,
        realized: parseFloat(program.realized_amount) || 0,
        company: program.company?.name
      }))
    }

    const client = getAIClient(user.ai_provider, user.ai_api_key)
    
    // Create AI prompt
    const prompt = `
Anda adalah seorang analis data yang ahli dalam menganalisis data anggaran perusahaan dan program RKAP (Rencana Kerja dan Anggaran Perusahaan).

Data yang tersedia:
${JSON.stringify(dataForAI, null, 2)}

Pertanyaan user: "${query}"

Berikan analisis langsung dalam format Markdown dengan struktur berikut (tanpa kalimat pembuka):

## 📊 Executive Summary
Berikan ringkasan singkat dan jelas dari analisis.

## 🔍 Key Insights
- Insight 1: [Deskripsi insight dengan data pendukung]
- Insight 2: [Deskripsi insight dengan data pendukung]
- Insight 3: [Deskripsi insight dengan data pendukung]
- Insight 4: [Deskripsi insight dengan data pendukung]

## 📈 Data Highlights
- **Total Anggaran**: Rp ${(dataForAI.summary.totalProposed / 1000000000).toFixed(1)} Miliar
- **Realisasi**: Rp ${(dataForAI.summary.totalRealized / 1000000000).toFixed(1)} Miliar
- **Tingkat Penyerapan**: ${dataForAI.summary.absorptionRate}%

## 💡 Recommendations
1. **Rekomendasi 1**: [Deskripsi rekomendasi spesifik]
2. **Rekomendasi 2**: [Deskripsi rekomendasi spesifik]
3. **Rekomendasi 3**: [Deskripsi rekomendasi spesifik]

Jawab dalam bahasa Indonesia dan berikan analisis yang mendalam berdasarkan data yang tersedia. Gunakan format markdown yang jelas dan mudah dibaca.
`

    let aiResponse
    const startTime = Date.now()
    
    try {
      if (user.ai_provider === 'gemini') {
        const model = client.getGenerativeModel({ model: 'gemini-2.0-flash-exp' })
        const result = await model.generateContent(prompt)
        aiResponse = result.response.text()
      } else if (user.ai_provider === 'openai' || user.ai_provider === 'deepseek') {
        const completion = await client.chat.completions.create({
          model: user.ai_provider === 'deepseek' ? 'deepseek-chat' : 'gpt-3.5-turbo',
          messages: [{ role: 'user', content: prompt }],
          max_tokens: 2000,
          temperature: 0.7
        })
        aiResponse = completion.choices[0].message.content
      }
    } catch (aiError) {
      console.error('❌ AI API Error:', aiError)
      console.error('❌ AI Error details:', {
        name: aiError.name,
        message: aiError.message,
        code: aiError.code
      })
      
      // Return error response instead of throwing
      return res.status(500).json({
        success: false,
        message: `AI API Error: ${aiError.message}`,
        error: aiError.message
      })
    }
    
    const responseTime = Date.now() - startTime
    
    console.log('🔍 Debug - AI Response received, length:', aiResponse?.length)
    console.log('🔍 Debug - Response time:', responseTime, 'ms')

    // Parse AI response
    const responseData = {
      id: Date.now(),
      query: query,
      summary: aiResponse.split('\n')[0] || aiResponse.substring(0, 200),
      insights: aiResponse.split('\n').filter(line => 
        line.includes('Insight') || line.includes('insight') || 
        line.includes('•') || line.includes('-')
      ).slice(0, 5),
      dataPoints: [
        { label: 'Total Anggaran', value: `Rp ${(dataForAI.summary.totalProposed / 1000000000).toFixed(1)} Miliar` },
        { label: 'Realisasi', value: `Rp ${(dataForAI.summary.totalRealized / 1000000000).toFixed(1)} Miliar` },
        { label: 'Penyerapan', value: `${dataForAI.summary.absorptionRate}%` }
      ],
      timestamp: new Date().toISOString(),
      rawResponse: aiResponse,
      markdownResponse: aiResponse // Keep original markdown for rendering
    }

    // Save AI response to database
    try {
      console.log('🔍 Debug - Attempting to save AI response to database')
      console.log('🔍 Debug - User ID:', user.id)
      console.log('🔍 Debug - Company ID:', user.company_id)
      console.log('🔍 Debug - Query length:', query.length)
      console.log('🔍 Debug - AI Response length:', aiResponse.length)
      
      const savedResponse = await AiResponse.create({
        user_id: user.id,
        company_id: user.company_id || 1, // Use default company_id 1 for superadmin
        query: query,
        summary: responseData.summary,
        raw_response: aiResponse,
        markdown_response: aiResponse,
        insights: JSON.stringify(responseData.insights),
        data_points: JSON.stringify(responseData.dataPoints),
        ai_provider: user.ai_provider,
        response_time_ms: responseTime,
        status: 'success'
      })
      
      responseData.id = savedResponse.id
      console.log('✅ AI response saved successfully with ID:', savedResponse.id)
    } catch (saveError) {
      console.error('❌ Error saving AI response:', saveError)
      console.error('❌ Error details:', {
        name: saveError.name,
        message: saveError.message,
        errors: saveError.errors
      })
      // Set error status but continue with response
      responseData.status = 'error'
      responseData.error = 'Failed to save response to database'
    }

    res.json({
      success: true,
      data: responseData
    })

  } catch (error) {
    console.error('AI analysis error:', error)
    res.status(500).json({
      success: false,
      message: 'AI analysis failed: ' + error.message
    })
  }
})

// Save AI response to database manually
router.post('/save-response', authenticateToken, async (req, res) => {
  try {
    console.log('🔍 Debug - Manual save endpoint hit')
    console.log('🔍 Debug - Request body:', req.body)
    
    const user = await User.findByPk(req.user.id, {
      include: [{ model: Company, as: 'company' }]
    })
    
    const {
      query,
      summary,
      raw_response,
      markdown_response,
      insights,
      data_points,
      ai_provider
    } = req.body
    
    const savedResponse = await AiResponse.create({
      user_id: user.id,
      company_id: user.company_id || 1, // Use default company_id 1 for superadmin
      query: query,
      summary: summary,
      raw_response: raw_response,
      markdown_response: markdown_response,
      insights: JSON.stringify(insights),
      data_points: JSON.stringify(data_points),
      ai_provider: ai_provider,
      response_time_ms: 0,
      status: 'success'
    })
    
    console.log('✅ Manual save successful with ID:', savedResponse.id)
    
    res.json({
      success: true,
      data: {
        id: savedResponse.id,
        message: 'Response saved successfully'
      }
    })
  } catch (error) {
    console.error('❌ Manual save error:', error)
    res.status(500).json({
      success: false,
      message: 'Failed to save response: ' + error.message
    })
  }
})

// Get AI insights for specific data type
router.get('/insights/:dataType', authenticateToken, async (req, res) => {
  try {
    const { dataType } = req.params
    const user = await User.findByPk(req.user.id)
    
    if (!user.ai_provider || !user.ai_api_key) {
      return res.status(400).json({
        success: false,
        message: 'AI provider not configured'
      })
    }

    // This would generate insights for specific data types
    // For now, return a placeholder
    res.json({
      success: true,
      insights: [
        'Terdapat peningkatan signifikan dalam kategori teknologi',
        'Perusahaan menunjukkan performa yang konsisten',
        'Realisasi anggaran mencapai target yang diharapkan'
      ]
    })

  } catch (error) {
    console.error('AI insights error:', error)
    res.status(500).json({
      success: false,
      message: 'Failed to generate insights: ' + error.message
    })
  }
})

// Get AI responses history
router.get('/responses', authenticateToken, async (req, res) => {
  try {
    const user = await User.findByPk(req.user.id)
    const { page = 1, limit = 20 } = req.query
    
    let whereClause = { user_id: user.id }
    if (user.role !== 'superadmin') {
      whereClause.company_id = user.company_id
    }

    const responses = await AiResponse.findAndCountAll({
      where: whereClause,
      order: [['createdAt', 'DESC']],
      limit: parseInt(limit),
      offset: (parseInt(page) - 1) * parseInt(limit),
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'email'] },
        { model: Company, as: 'company', attributes: ['id', 'name'] }
      ]
    })

    res.json({
      success: true,
      data: {
        responses: responses.rows.map(response => ({
          id: response.id,
          query: response.query,
          summary: response.summary,
          rawResponse: response.raw_response,
          markdownResponse: response.markdown_response,
          insights: response.insights,
          dataPoints: response.data_points,
          aiProvider: response.ai_provider,
          timestamp: response.createdAt,
          status: response.status
        })),
        pagination: {
          total: responses.count,
          page: parseInt(page),
          limit: parseInt(limit),
          totalPages: Math.ceil(responses.count / parseInt(limit))
        }
      }
    })

  } catch (error) {
    console.error('Get AI responses error:', error)
    res.status(500).json({
      success: false,
      message: 'Failed to get AI responses: ' + error.message
    })
  }
})

// Delete AI response
router.delete('/responses/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params
    const user = await User.findByPk(req.user.id)
    
    let whereClause = { id, user_id: user.id }
    if (user.role !== 'superadmin') {
      whereClause.company_id = user.company_id
    }

    const response = await AiResponse.findOne({ where: whereClause })
    if (!response) {
      return res.status(404).json({
        success: false,
        message: 'AI response not found'
      })
    }

    await response.destroy()

    res.json({
      success: true,
      message: 'AI response deleted successfully'
    })

  } catch (error) {
    console.error('Delete AI response error:', error)
    res.status(500).json({
      success: false,
      message: 'Failed to delete AI response: ' + error.message
    })
  }
})

// Generate AI report
router.post('/generate-report', authenticateToken, async (req, res) => {
  try {
    const { reportType, filters } = req.body
    const user = await User.findByPk(req.user.id)
    
    if (!user.ai_provider || !user.ai_api_key) {
      return res.status(400).json({
        success: false,
        message: 'AI provider not configured'
      })
    }

    // This would generate comprehensive reports
    // For now, return a placeholder
    res.json({
      success: true,
      report: {
        title: `${reportType} Report`,
        generatedAt: new Date().toISOString(),
        summary: 'AI-generated report based on your data'
      }
    })

  } catch (error) {
    console.error('AI report generation error:', error)
    res.status(500).json({
      success: false,
      message: 'Failed to generate report: ' + error.message
    })
  }
})

module.exports = router
