const { SessionLog, ApiLog } = require('../models');

// Session logging middleware
const sessionLogger = (req, res, next) => {
  const startTime = Date.now();
  const originalSend = res.send;
  
  // Get user info if available
  const userId = req.user?.id || null;
  const companyId = req.user?.company_id || null;
  
  // Get client info
  const ipAddress = req.ip || req.connection.remoteAddress || req.socket.remoteAddress || 
                   (req.connection.socket ? req.connection.socket.remoteAddress : null) ||
                   req.headers['x-forwarded-for']?.split(',')[0] || 'unknown';
  const userAgent = req.headers['user-agent'] || 'unknown';
  
  // Override res.send to capture response
  res.send = function(data) {
    const responseTime = Date.now() - startTime;
    const statusCode = res.statusCode;
    const success = statusCode >= 200 && statusCode < 400;
    
    // Log session data asynchronously (only for authenticated requests)
    if (userId && req.path !== '/health') {
      setImmediate(async () => {
        try {
          await SessionLog.create({
            user_id: userId,
            company_id: companyId,
            session_token: req.headers.authorization?.split(' ')[1] || null,
            login_time: new Date(),
            logout_time: null,
            ip_address: ipAddress,
            user_agent: userAgent,
            success: success,
            failure_reason: success ? null : `HTTP ${statusCode}`
          });
        } catch (error) {
          console.error('Session logging error:', error);
        }
      });
    }
    
    // Call original send
    originalSend.call(this, data);
  };
  
  next();
};

// API logging middleware
const apiLogger = (req, res, next) => {
  const startTime = Date.now();
  const originalSend = res.send;
  
  // Get user info if available
  const userId = req.user?.id || null;
  const companyId = req.user?.company_id || null;
  
  // Get client info
  const ipAddress = req.ip || req.connection.remoteAddress || req.socket.remoteAddress || 
                   (req.connection.socket ? req.connection.socket.remoteAddress : null) ||
                   req.headers['x-forwarded-for']?.split(',')[0] || 'unknown';
  const userAgent = req.headers['user-agent'] || 'unknown';
  
  // Override res.send to capture response
  res.send = function(data) {
    const responseTime = Date.now() - startTime;
    const statusCode = res.statusCode;
    
    // Log API data asynchronously (always log, but handle null user_id)
    setImmediate(async () => {
      try {
        await ApiLog.create({
          user_id: userId, // Can be null for unauthenticated requests
          company_id: companyId,
          endpoint: req.originalUrl || req.url,
          method: req.method,
          status_code: statusCode,
          response_time_ms: responseTime,
          request_data: req.method !== 'GET' ? JSON.stringify(req.body) : null,
          response_data: statusCode >= 400 ? JSON.stringify(data) : null,
          ip_address: ipAddress,
          user_agent: userAgent,
          timestamp: new Date()
        });
      } catch (error) {
        console.error('API logging error:', error);
      }
    });
    
    // Call original send
    originalSend.call(this, data);
  };
  
  next();
};

module.exports = {
  sessionLogger,
  apiLogger
};
