const jwt = require('jsonwebtoken');
const { User } = require('../models');

const authenticateToken = async (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.status(401).json({ message: 'Access token required' });
  }

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const user = await User.findByPk(decoded.userId, {
      include: [{ model: require('../models').Company, as: 'company' }]
    });

    if (!user) {
      return res.status(401).json({ message: 'Invalid token' });
    }

    req.user = user;
    next();
  } catch (error) {
    return res.status(403).json({ message: 'Invalid or expired token' });
  }
};

const authorizeRoles = (...roles) => {
  return (req, res, next) => {
    if (!req.user) {
      return res.status(401).json({ message: 'Authentication required' });
    }

    if (!roles.includes(req.user.role)) {
      return res.status(403).json({ message: 'Insufficient permissions' });
    }

    next();
  };
};

const authorizeCompany = (req, res, next) => {
  if (!req.user) {
    return res.status(401).json({ message: 'Authentication required' });
  }

  // Superadmin can access all companies
  if (req.user.role === 'superadmin') {
    return next();
  }

  // Admin and User can only access their own company
  const companyId = req.params.companyId || req.body.company_id || req.query.company_id;
  
  if (companyId && req.user.company_id !== parseInt(companyId)) {
    return res.status(403).json({ message: 'Access denied to this company' });
  }

  next();
};

module.exports = {
  authenticateToken,
  authorizeRoles,
  authorizeCompany
};
